/*
  This file is part of CDO. CDO is a collection of Operators to manipulate and analyse Climate model Data.

  Author: Uwe Schulzweida

*/

/*
   This module contains the following operators:

      Pardup     pardup          Duplicate parameters
      Pardup     parmul          Multiply parameters
*/

#include <cdi.h>

#include "cdo_vlist.h"
#include "process_int.h"
#include "param_conversion.h"

class Pardup : public Process
{
public:
  using Process::Process;
  inline static CdoModule module = {
    .name = "Pardup",
    .operators = { { "pardup"}, { "parmul"} },
    .aliases = {},
    .mode = EXPOSED,     // Module mode: 0:intern 1:extern
    .number = CDI_REAL,  // Allowed number type
    .constraints = { 1, 1, NoRestriction },
  };
  inline static RegisterEntry<Pardup> registration = RegisterEntry<Pardup>(module);
  int PARDUP, PARMUL;
  CdoStreamID streamID1;
  CdoStreamID streamID2;

  int taxisID1;
  int taxisID2;
  int nmul = 0;
  int nvars;

  VarList varList1;
  std::vector<RecordInfo> recList;
  Varray<double> array;
  Varray2D<double> vardata;
  std::vector<std::vector<size_t>> varnumMissVals;

public:
  void
  init()
  {

    PARDUP = module.get_id("pardup");
    PARMUL = module.get_id("parmul");

    const auto operatorID = cdo_operator_id();

    if (operatorID == PARDUP) { nmul = 2; }
    else if (operatorID == PARMUL)
      {
        operator_input_arg("number of multiply");
        nmul = parameter_to_int(cdo_operator_argv(0));
      }
    else
      cdo_abort("operator not implemented!");

    streamID1 = cdo_open_read(0);

    const auto vlistID1 = cdo_stream_inq_vlist(streamID1);
    const auto vlistID2 = vlistDuplicate(vlistID1);

    taxisID1 = vlistInqTaxis(vlistID1);
    taxisID2 = taxisDuplicate(taxisID1);
    vlistDefTaxis(vlistID2, taxisID2);

    varList_init(varList1, vlistID1);

    nvars = vlistNvars(vlistID1);

    const auto maxrecs = vlistNrecs(vlistID1);
    recList = std::vector<RecordInfo>(maxrecs);

    const auto gridsizemax = vlistGridsizeMax(vlistID1);
    array = Varray<double>(gridsizemax);
    vardata = Varray2D<double>(nvars);
    varnumMissVals = std::vector<std::vector<size_t>>(nvars);

    for (int varID = 0; varID < nvars; ++varID)
      {
        const auto gridsize = varList1[varID].gridsize;
        const auto nlevels = varList1[varID].nlevels;
        vardata[varID].resize(gridsize * nlevels);
        varnumMissVals[varID].resize(nlevels);
      }

    for (int i = 1; i < nmul; ++i)
      {
        vlistCat(vlistID2, vlistID1);
        for (int varID = 0; varID < nvars; ++varID)
          vlistDefVarParam(vlistID2, varID + nvars * i, cdiEncodeParam(-(varID + nvars * i + 1), 255, 255));
      }

    streamID2 = cdo_open_write(1);
    cdo_def_vlist(streamID2, vlistID2);
  }
  void
  run()
  {
    int tsID = 0;
    while (true)
      {
        const auto nrecs = cdo_stream_inq_timestep(streamID1, tsID);
        if (nrecs == 0) break;

        cdo_taxis_copy_timestep(taxisID2, taxisID1);
        cdo_def_timestep(streamID2, tsID);

        for (int recID = 0; recID < nrecs; ++recID)
          {
            int varID, levelID;
            cdo_inq_record(streamID1, &varID, &levelID);

            recList[recID].set(varID, levelID);

            const auto gridsize = varList1[varID].gridsize;
            const auto offset = gridsize * levelID;
            auto single = &vardata[varID][offset];

            size_t numMissVals;
            cdo_read_record(streamID1, single, &numMissVals);
            varnumMissVals[varID][levelID] = numMissVals;
          }

        for (int i = 0; i < nmul; ++i)
          for (int recID = 0; recID < nrecs; ++recID)
            {
              auto [varID, levelID] = recList[recID].get();

              const auto varID2 = varID + i * nvars;

              const auto gridsize = varList1[varID].gridsize;
              const auto offset = gridsize * levelID;
              auto single = &vardata[varID][offset];
              const auto numMissVals = varnumMissVals[varID][levelID];

              array_copy(gridsize, single, array.data());
              cdo_def_record(streamID2, varID2, levelID);
              cdo_write_record(streamID2, array.data(), numMissVals);
            }

        tsID++;
      }
  }

  void
  close()
  {
    cdo_stream_close(streamID2);
    cdo_stream_close(streamID1);
  }
};
