/*
 * Copyright (c) 2024
 *      Tim Woodall. All rights reserved
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 * 1. Redistributions of source code must retain the above copyright notice,
 *    this list of conditions and the following disclaimer.
 *
 * 2. Redistributions in binary form must reproduce the above copyright notice,
 *    this list of conditions and the following disclaimer in the documentation
 *    and/or other materials provided with the distribution.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 *
 * SPDX short identifier: BSD-2-Clause
 */

#include <config.h>
#include "bswap_header.h"

#include <inttypes.h>
#include <cstddef>
#include <cstring>
#include <algorithm>
#include <iostream>
#include <err.h>
#include <fcntl.h>
#include <sys/stat.h>

#define CHECKSUM 84446

void swap1(int count, uint8_t*& data) {
	data += count;
}

void swap2(int count, uint8_t*& data) {
	while(count--) {
		std::swap(data[0], data[1]);
		data += 2;
	}
}

void swap4(int count, uint8_t*& data) {
	while(count--) {
		std::swap(data[0], data[3]);
		std::swap(data[1], data[2]);
		data += 4;
	}
}

void swap8(int count, uint8_t*& data) {
	while(count--) {
		std::swap(data[0], data[7]);
		std::swap(data[1], data[6]);
		std::swap(data[2], data[5]);
		std::swap(data[3], data[4]);
		data += 8;
	}
}

uint32_t get_s_addr_length(const uint8_t* ptr) {
	/*
	 * 1 - 64 encoded as 0-63 in n V[0] = (0nnnnnnf) (f is the data/sparse flag)
	 * if ( V[0] < 128 ) return (*V >> 1) + 1 )
	 * 65 - 16384 encoded as 0-16319 in Nn V[0] = (1nnnnnnf) V[1] = (NNNNNNNN)
	 * if (V[0] >= 128) return V[1]*64 + ((V[0]>>1)&63) + 65;
	 * 128 values left unused for future special cases (V[1] = 255) 1nnnnnnn 11111111
	 * if get_s_addr_length() >=65 then length is encoded in two bytes else in one.
	 */
	if (*ptr < 128)
		return (*ptr>>1)+1;
	return *(ptr+1)*64 + ((*ptr>>1)&63) + 65;
}

void do_bswap_hdr(header* hdr) {
	/**
	 * We are byteswapping a u_spcl.s_spcl here from dumprestore.h
	8 <<	swap 8 terms
		i <<	c_type
			c_date
			c_ddate
			c_volume
			c_tapea_lo
			c_inumber
			c_magic
			c_checksum
	4 <<	swap 4 terms
		s	di_mode		(new_bsd_inode)
			di_nlink
			oldids[2]
	1 <<    swap 1 term
		l	di_size
	29 <<	swap 29 terms
		i	di_atime	0
			di_mtime	2
			di_ctime	4
			di_db[NDADDR]	6
			di_ib[NIADDR]	18
			di_flags	21
			di_blocks	22
			di_gen		23
			di_uid		24
			di_gid		25
			di_spare[2]	26	(end of new_bsd_inode)
			c_count		29
	528 <<	swap 528 terms
		b	c_data	0
			c_label	512
	i <<	Swap 1 term
			c_level
	192 <<	swap 192 terms
		b <<    c_filesys	0
			c_dev		64
			c_host		128
	6 <<	swap 6 terms
		i <<	c_flags
			c_firstrec
			c_ntrec
			c_extattributes
			c_tapea_hi
			c_firstrec_hi
	c_spare[28] not swapped.
	*/

	uint8_t* data = reinterpret_cast<uint8_t*>(hdr);
	swap4(8, data);		//32	32
	swap2(4, data);		//8	40
	swap8(1, data);		//8	48
	swap4(29, data);	//116	164
	swap1(528, data);	//528	692
	swap4(1, data);		//4	696
	swap1(192, data);	//192	888
	swap4(6, data);		//24	912
				//112	1024
};

bool checksum(header* hdr) {
	bool ok = false;
	bool bswap = false;
	if (hdr->c_magic != NFS_MAGIC) {
		auto* data = reinterpret_cast<uint8_t*>(hdr);
		swap4(sizeof *hdr / sizeof(uint32_t), data);
		bswap = true;
	}
	if (hdr->c_magic == NFS_MAGIC) {
		auto* lp = reinterpret_cast<const uint32_t *>(hdr);
		uint32_t sum = 0;
		uint32_t cnt = sizeof *hdr / sizeof *lp;
		while (cnt--)
			sum += *lp++;
		ok = sum == CHECKSUM;
	}
	if(bswap) {
		auto* data = reinterpret_cast<uint8_t*>(hdr);
		swap4(sizeof *hdr / sizeof(uint32_t), data);
	}
	return ok;
}

void setchecksum(header* hdr) {
	bool bswap = false;
	if (hdr->c_magic != NFS_MAGIC) {
		auto* data = reinterpret_cast<uint8_t*>(hdr);
		swap4(sizeof *hdr / sizeof(uint32_t), data);
		bswap = true;
	}
	if (hdr->c_magic == NFS_MAGIC) {
		uint32_t* lp = reinterpret_cast<uint32_t *>(hdr);
		uint32_t sum = 0;
		uint32_t cnt = sizeof *hdr / sizeof *lp;
		hdr->c_checksum = 0;
		while (cnt--)
			sum += *lp++;
		hdr->c_checksum = (uint32_t)CHECKSUM - sum;
	}
	if(bswap) {
		auto* data = reinterpret_cast<uint8_t*>(hdr);
		swap4(sizeof *hdr / sizeof(uint32_t), data);
	}
}

void bswap_header(uint8_t* data, size_t bytes, bswap_context* ctx) {
	if(!data || !bytes) {
		if(ctx->seen_end) {
			ctx->subcount = 0;
			ctx->state = 0;
			ctx->count = 0;
			ctx->inode_is_directory = false;
			ctx->inode_data_size = 0;
		} else
			ctx->new_vol = true;
		return;
	}
	if (bytes % sizeof(header))
		errx(1, "Buffer is not a multiple of tape blocks in size");
	header* hdr = reinterpret_cast<header*>(data);
	bytes /= sizeof *hdr;

	auto skip_holes = [&ctx]() {
		while (ctx->count && !(*ctx->s_addr_ptr&1)) {
			if(get_s_addr_length(ctx->s_addr_ptr) > 64)
				ctx->s_addr_ptr += 2;
			else
				ctx->s_addr_ptr++;
			ctx->count--;
		}
		if (!ctx->count) {
			ctx->subcount = 0;
			ctx->state = 0;
		} else
			ctx->subcount = get_s_addr_length(ctx->s_addr_ptr);
	};

	auto do_bswap_direct = [&]() {
		uint8_t* data = reinterpret_cast<uint8_t*>(hdr);
		size_t consumed = 0;
		const direct* ptr = reinterpret_cast<const direct*>(data);
		while(consumed < sizeof *hdr && ctx->inode_data_size) {
			if (ctx->to_cpu) {
				swap4(1, data);
				swap2(1, data);
				data -= 6;
			}
			uint16_t reclen = ptr->d_reclen;
			if (ctx->from_cpu) {
				swap4(1, data);
				swap2(1, data);
				data -= 6;
			}
			data += reclen;
			consumed += reclen;
			ctx->inode_data_size -= reclen;
			ptr = reinterpret_cast<const direct*>(data);
		}
	};

	auto process_block = [&]() {
		switch(ctx->state) {
			case 0: {
				if (!checksum(hdr)) {
					std::cerr << "SKIPPING UNKNOWN" << std::endl;
					hdr++;
					bytes--;
					break;
				}
				if (ctx->to_cpu)
					do_bswap_hdr(hdr);
				if (hdr->c_magic != NFS_MAGIC) {
					std::cerr << "SKIPPING UNKNOWN" << std::endl;
					hdr++;
					bytes--;
					break;
				}
#if 0
				ctx->subcount = 0;
				ctx->count = 0;
				ctx->inode_is_directory = false;
				ctx->inode_data_size = 0;
#endif
				ctx->seen_end = false;
				switch (hdr->c_type) {
					case TS_END:
						ctx->seen_end = true;
						[[fallthrough]];
					case TS_TAPE: {
						uint8_t* data = reinterpret_cast<uint8_t*>(hdr->c_data.s_inos);
						if (ctx->to_cpu || ctx->from_cpu)
							swap4(128, data);
						break;
					}
					case TS_CLRI:
						ctx->count = hdr->c_count;
						ctx->state = ctx->count?TS_CLRI:0;
						break;
					case TS_BITS:
						ctx->count = hdr->c_count;
						ctx->state = ctx->count?TS_BITS:0;
						break;
					case TS_INODE:
						ctx->inode_is_directory = !(hdr->c_flags & DR_EXTATTRIBUTES) && (hdr->c_dinode.di_mode & S_IFMT) == S_IFDIR;
						ctx->inode_data_size = hdr->c_dinode.di_size;
						[[fallthrough]];
					case TS_ADDR:
						ctx->count = hdr->c_count;
						/* TODO - there was a problem with EAs sometimes not having s_addr populated properly */
						std::memcpy(ctx->s_addr, hdr->c_data.s_addrs, 512);
						ctx->s_addr_ptr = ctx->s_addr;
						skip_holes();
						if (ctx->count)
							ctx->state = TS_INODE;
						break;
					default:
						errx(1, "unknown state on tape");
						break;
				}
				if (ctx->from_cpu)
					do_bswap_hdr(hdr);
				setchecksum(hdr);
				hdr++;
				bytes--;
				break;
			}
			case TS_INODE:
				if (ctx->inode_is_directory)
					do_bswap_direct();
				hdr++;
				bytes--;
				if(--ctx->subcount == 0) {
					if(--ctx->count == 0)
						ctx->state = 0;
					else {
						if(get_s_addr_length(ctx->s_addr_ptr) > 64)
							ctx->s_addr_ptr += 2;
						else
							ctx->s_addr_ptr++;
					}
					skip_holes();
				}
				break;
			case TS_BITS:
			case TS_CLRI:
				hdr++;
				bytes--;
				if(--ctx->count == 0)
					ctx->state = 0;
				break;
			default:
				errx(1, "unexpected state while reading tape");
				break;
		}
	};
	if (ctx->new_vol) {
		auto save_hdr = hdr;
		int saved_state = ctx->state;
		ctx->state = 0;
		if (ctx->to_cpu)
			process_block();
		if (save_hdr->c_magic != NFS_MAGIC || !checksum(save_hdr) || save_hdr->c_type != TS_TAPE) {
			std::cerr << "EXPECTED TS_TAPE at start of volume save_hdr->c_magic=" << std::hex << save_hdr->c_magic << std::endl;
		} else if (!ctx->to_cpu)
			process_block();
		ctx->state = saved_state;
		ctx->new_vol = false;
	}
	while (bytes)
		process_block();
}

/* vim: set sw=8 sts=8 ts=8 noexpandtab: */
