import textwrap
from typing import List, TYPE_CHECKING, Sequence

import pytest

from debputy.filesystem_scan import build_virtual_fs
from debputy.lsp.config.config_options import DCO_SPELLCHECK_COMMENTS
from debputy.lsp.languages.lsp_debian_control import _reformat_debian_control
from debputy.lsp.lsp_debian_control_reference_data import CURRENT_STANDARDS_VERSION
from debputy.lsp.maint_prefs import MaintainerPreferenceTable
from debputy.lsp.text_edit import apply_text_edits
from debputy.lsp.vendoring._deb822_repro import parse_deb822_file
from debputy.lsprotocol.types import DiagnosticSeverity
from debputy.packages import DctrlParser
from debputy.plugin.api import virtual_path_def
from debputy.plugin.api.feature_set import PluginProvidedFeatureSet
from lint_tests.lint_tutil import (
    group_diagnostics_by_severity,
    requires_levenshtein,
    LintWrapper,
    diag_range_to_text,
    ReformatWrapper,
)
from tutil import build_time_only, with_plugins_loaded


if TYPE_CHECKING:
    import lsprotocol.types as types
else:
    import debputy.lsprotocol.types as types


ALL_REQUIRED_FILES = [
    "debian/changelog",
    "debian/copyright",
]


@pytest.fixture
def reformater(
    debputy_plugin_feature_set: PluginProvidedFeatureSet,
    lint_dctrl_parser: DctrlParser,
    maintainer_preference_table: MaintainerPreferenceTable,
) -> ReformatWrapper:

    return ReformatWrapper(
        "/nowhere/debian/control",
        _reformat_debian_control,
        debputy_plugin_feature_set,
        lint_dctrl_parser,
        maintainer_preference_table,
    )


def apply_formatting_edits(
    lines: List[str],
    edits: Sequence[types.TextEdit],
) -> str:
    orig = "".join(lines)
    return apply_text_edits(orig, lines, edits)


def test_dctrl_reformat(reformater: ReformatWrapper) -> None:
    lines = textwrap.dedent(
        """\
    Source: foo
    Some-Other-Field: bar
    Build-Depends: debhelper-compat (= 13)
    Homepage: https://example.com

    Package: foo
    Architecture: all
    Section: devel
    """
    ).splitlines(keepends=True)

    edits = reformater.reformat(lines)
    # By default, we do nothing
    assert not edits

    edits_black = reformater.reformat_with_named_style("black", lines)
    assert edits_black

    actual_reformatted_black = apply_formatting_edits(lines, edits_black)

    expected_reformatted_black = textwrap.dedent(
        """\
    Source: foo
    Some-Other-Field: bar
    Build-Depends:
     debhelper-compat (= 13),
    Homepage: https://example.com

    Package: foo
    Architecture: all
    Section: devel
    """
    )

    assert actual_reformatted_black == expected_reformatted_black


def test_dctrl_reformat_canonicalize_field_names(reformater: ReformatWrapper) -> None:
    lines = textwrap.dedent(
        """\
    source: foo
    # This field will not change case, since it is unknown
    some-other-field: bar
    BUILD-DEPENDS:
     debhelper-compat (= 13),
    # In memory of Steve (#1068220)
    HoMEpaGe: https://example.com

    package: foo
    architecture: all
    Section: devel
    """
    ).splitlines(keepends=True)

    edits = reformater.reformat(lines)
    # By default, we do nothing
    assert not edits

    edits_black = reformater.reformat_with_named_style("black", lines)
    assert edits_black

    actual_reformatted_black = apply_formatting_edits(lines, edits_black)

    expected_reformatted_black = textwrap.dedent(
        """\
    Source: foo
    # This field will not change case, since it is unknown
    some-other-field: bar
    Build-Depends:
     debhelper-compat (= 13),
    # In memory of Steve (#1068220)
    Homepage: https://example.com

    Package: foo
    Architecture: all
    Section: devel
    """
    )

    assert actual_reformatted_black == expected_reformatted_black
