/*
 * TagLib C# bindings
 * by Clemens Koppensteiner and Brad Taylor
 *
 * This work is hereby released into the Public Domain. To view a copy of the
 * public domain dedication, visit
 * http://creativecommons.org/licenses/publicdomain/ or send a letter to
 * Creative Commons, 543 Howard Street, 5th Floor, San Francisco, California,
 * 94105, USA.
 */

using System;
using System.Runtime.InteropServices;

namespace TagLib
{
	public class File : Object, IDisposable
	{
		public readonly string Name;
		
		public File(string filename) : base (IntPtr.Zero) {    	
			/*
			 * XXX: Hacky, yet approved, fix for #59:
			 * "When ISO8859-1 Characters Attack!"
			 */
			try {
				taglibsharpglue_set_encoding_utf16();
			} catch (DllNotFoundException) {
				Console.WriteLine ("WARNING: Unable to set default encoding to UTF-16."
				                   + "Some Unicode tag-writing operations may fail or produce unexpected results.");
			}

			if (!System.IO.File.Exists(filename)) {
				throw new System.IO.FileNotFoundException(String.Format("Could not find file \"{0}\"", filename)); 
			}
			
			ptr = taglib_file_new(filename);
			if (ptr == IntPtr.Zero) {
				throw new ApplicationException("taglib: Got null pointer when trying to open file: " + filename);
			}
			Name = filename;
		}
		
		protected File(IntPtr ptr) : base (ptr) {}
		
		~File()
		{
			doDispose();
		}
		
		void doDispose()
		{
			taglib_file_free(ptr);
			ptr = IntPtr.Zero;
		}
		
		public void Dispose()
		{
			doDispose();
			GC.SuppressFinalize(this);
		}
		
		Tag tag;
		public Tag Tag {
			get {
				if (tag == null) {
					IntPtr tagptr = taglib_file_tag(ptr);
					if (tagptr == IntPtr.Zero) {
						throw new ApplicationException("taglib: Got null pointer when trying to get tag of file " + Name);
					}
					tag = new Tag(tagptr);
				}
				
				return tag;
			}
		}
		
		AudioProperties audioProperties;
		public AudioProperties AudioProperties {
			get {
				if (audioProperties == null) {
					IntPtr apptr = taglib_file_audioproperties(ptr);
					if (apptr == IntPtr.Zero) {
						throw new ApplicationException("taglib: Got null pointer when trying to get audioproperties of file: " + Name);
					}
					audioProperties = new AudioProperties(apptr);
				}
				
				return audioProperties;
			}
		}
		
		public void Save()
		{
			taglib_file_save(ptr);
		}
		
		[DllImport("libtag_c")]
		static extern IntPtr taglib_file_new(string filename); 
		
		[DllImport("libtag_c")]
		static extern void taglib_file_free(IntPtr file);
		
		[DllImport("libtag_c")]
		static extern IntPtr taglib_file_tag(IntPtr file);
		
		[DllImport("libtag_c")]
		static extern IntPtr taglib_file_audioproperties(IntPtr file);
		
		[DllImport("libtag_c")]
		static extern void taglib_file_save(IntPtr file);

		[DllImport("taglibsharpglue")]
		static extern void taglibsharpglue_set_encoding_utf16();
	}
}
