;;;; -*- mode: lisp; indent-tabs-mode: nil -*-
;;;; threefish.lisp -- implementation of the Threefish block cipher

(in-package :crypto)


;;; Constants

(defconstant +threefish-key-schedule-constant+ #x1bd11bdaa9fc1a22)


;;; Functions common to all the block sizes

(defun threefish-set-key (cipher key-data)
  (let* ((key-words (/ (block-length cipher) 8))
         (key (make-array (1+ key-words)
                          :element-type '(unsigned-byte 64)
                          :initial-element 0))
         (parity +threefish-key-schedule-constant+))
    (dotimes (i key-words)
      (let ((n (ub64ref/le key-data (* 8 i))))
        (setf (aref key i) n)
        (setf parity (logxor parity n))))
    (setf (aref key key-words) parity)
    (setf (threefish-key cipher) key)
    (values)))

(defun threefish-set-tweak (cipher tweak-data)
  (unless (= (length tweak-data) 16)
    (error 'ironclad-error :format-control "The tweak size must be 16 bytes."))
  (let ((tweak (threefish-tweak cipher)))
    (setf (aref tweak 0) (ub64ref/le tweak-data 0)
          (aref tweak 1) (ub64ref/le tweak-data 8)
          (aref tweak 2) (logxor (aref tweak 0) (aref tweak 1)))))


;;; Implementation for blocks of 256 bits

(defclass threefish256 (cipher 32-byte-block-mixin)
  ((key :accessor threefish-key
        :type (simple-array (unsigned-byte 64) (5)))
   (tweak :accessor threefish-tweak
          :initform (make-array 3
                                :element-type '(unsigned-byte 64)
                                :initial-contents '(0 0 0))
          :type (simple-array (unsigned-byte 64) (3)))))

(defmethod shared-initialize :after ((cipher threefish256) slot-names
                                     &rest initargs
                                     &key (tweak nil tweak-p)
                                     &allow-other-keys)
  (when tweak
    (threefish-set-tweak cipher tweak))
  cipher)

(defmethod schedule-key ((cipher threefish256) key)
  (threefish-set-key cipher key)
  cipher)

(define-block-encryptor threefish256 32
  (let* ((key (threefish-key context))
         (tweak (threefish-tweak context))
         (k0 (aref key 0))
         (k1 (aref key 1))
         (k2 (aref key 2))
         (k3 (aref key 3))
         (k4 (aref key 4))
         (t0 (aref tweak 0))
         (t1 (aref tweak 1))
         (t2 (aref tweak 2)))
    (declare (type (simple-array (unsigned-byte 64) (5)) key)
             (type (simple-array (unsigned-byte 64) (3)) tweak)
             (type (unsigned-byte 64) k0 k1 k2 k3 k4 t0 t1 t2))
    (with-words ((b0 b1 b2 b3) plaintext plaintext-start :big-endian nil :size 8)
      (macrolet ((arx (x0 x1 x2 x3 x4 x5 x6 x7
                       x8 x9 x10 x11 x12 x13 x14 x15)
                   `(setf b1 (mod64+ b1 (mod64+ ,x0 ,x1))
                          b0 (mod64+ b0 (mod64+ ,x2 ,x3))
                          b1 (logxor (rol64 b1 14) b0)
                          b3 (mod64+ b3 (mod64+ ,x4 ,x5))
                          b2 (mod64+ b2 (mod64+ b3 (mod64+ ,x6 ,x7)))
                          b3 (logxor (rol64 b3 16) b2)
                          b0 (mod64+ b0 b3)
                          b3 (logxor (rol64 b3 52) b0)
                          b2 (mod64+ b2 b1)
                          b1 (logxor (rol64 b1 57) b2)
                          b0 (mod64+ b0 b1)
                          b1 (logxor (rol64 b1 23) b0)
                          b2 (mod64+ b2 b3)
                          b3 (logxor (rol64 b3 40) b2)
                          b0 (mod64+ b0 b3)
                          b3 (logxor (rol64 b3 5) b0)
                          b2 (mod64+ b2 b1)
                          b1 (logxor (rol64 b1 37) b2)
                          b1 (mod64+ b1 (mod64+ ,x8 ,x9))
                          b0 (mod64+ b0 (mod64+ ,x10 ,x11))
                          b1 (logxor (rol64 b1 25) b0)
                          b3 (mod64+ b3 (mod64+ ,x12 ,x13))
                          b2 (mod64+ b2 (mod64+ b3 (mod64+ ,x14 ,x15)))
                          b3 (logxor (rol64 b3 33) b2)
                          b0 (mod64+ b0 b3)
                          b3 (logxor (rol64 b3 46) b0)
                          b2 (mod64+ b2 b1)
                          b1 (logxor (rol64 b1 12) b2)
                          b0 (mod64+ b0 b1)
                          b1 (logxor (rol64 b1 58) b0)
                          b2 (mod64+ b2 b3)
                          b3 (logxor (rol64 b3 22) b2)
                          b0 (mod64+ b0 b3)
                          b3 (logxor (rol64 b3 32) b0)
                          b2 (mod64+ b2 b1)
                          b1 (logxor (rol64 b1 32) b2))))
        (arx k1 t0 b1 k0 k3 0 k2 t1
             k2 t1 b1 k1 k4 1 k3 t2)
        (arx k3 t2 b1 k2 k0 2 k4 t0
             k4 t0 b1 k3 k1 3 k0 t1)
        (arx k0 t1 b1 k4 k2 4 k1 t2
             k1 t2 b1 k0 k3 5 k2 t0)
        (arx k2 t0 b1 k1 k4 6 k3 t1
             k3 t1 b1 k2 k0 7 k4 t2)
        (arx k4 t2 b1 k3 k1 8 k0 t0
             k0 t0 b1 k4 k2 9 k1 t1)
        (arx k1 t1 b1 k0 k3 10 k2 t2
             k2 t2 b1 k1 k4 11 k3 t0)
        (arx k3 t0 b1 k2 k0 12 k4 t1
             k4 t1 b1 k3 k1 13 k0 t2)
        (arx k0 t2 b1 k4 k2 14 k1 t0
             k1 t0 b1 k0 k3 15 k2 t1)
        (arx k2 t1 b1 k1 k4 16 k3 t2
             k3 t2 b1 k2 k0 17 k4 t0))
      (setf b0 (mod64+ b0 k3)
            b1 (mod64+ b1 (mod64+ k4 t0))
            b2 (mod64+ b2 (mod64+ k0 t1))
            b3 (mod64+ b3 (mod64+ k1 18)))
      (store-words ciphertext ciphertext-start b0 b1 b2 b3)
      (values))))

(define-block-decryptor threefish256 32
  (let* ((key (threefish-key context))
         (tweak (threefish-tweak context))
         (k0 (aref key 0))
         (k1 (aref key 1))
         (k2 (aref key 2))
         (k3 (aref key 3))
         (k4 (aref key 4))
         (t0 (aref tweak 0))
         (t1 (aref tweak 1))
         (t2 (aref tweak 2)))
    (declare (type (simple-array (unsigned-byte 64) (5)) key)
             (type (simple-array (unsigned-byte 64) (3)) tweak)
             (type (unsigned-byte 64) k0 k1 k2 k3 k4 t0 t1 t2))
    (with-words ((b0 b1 b2 b3) ciphertext ciphertext-start :big-endian nil :size 8)
      (setf b0 (mod64- b0 k3)
            b1 (mod64- b1 (mod64+ k4 t0))
            b2 (mod64- b2 (mod64+ k0 t1))
            b3 (mod64- b3 (mod64+ k1 18)))
      (macrolet ((arx (x0 x1 x2 x3 x4 x5 x6
                       x7 x8 x9 x10 x11 x12 x13)
                   `(setf b3 (ror64 (logxor b3 b0) 32)
                          b0 (mod64- b0 b3)
                          b1 (ror64 (logxor b1 b2) 32)
                          b2 (mod64- b2 b1)
                          b1 (ror64 (logxor b1 b0) 58)
                          b0 (mod64- b0 b1)
                          b3 (ror64 (logxor b3 b2) 22)
                          b2 (mod64- b2 b3)
                          b3 (ror64 (logxor b3 b0) 46)
                          b0 (mod64- b0 b3)
                          b1 (ror64 (logxor b1 b2) 12)
                          b2 (mod64- b2 b1)
                          b1 (ror64 (logxor b1 b0) 25)
                          b0 (mod64- b0 (mod64+ b1 ,x0))
                          b1 (mod64- b1 (mod64+ ,x1 ,x2))
                          b3 (ror64 (logxor b3 b2) 33)
                          b2 (mod64- b2 (mod64+ b3 (mod64+ ,x3 ,x4)))
                          b3 (mod64- b3 (mod64+ ,x5 ,x6))
                          b3 (ror64 (logxor b3 b0) 5)
                          b0 (mod64- b0 b3)
                          b1 (ror64 (logxor b1 b2) 37)
                          b2 (mod64- b2 b1)
                          b1 (ror64 (logxor b1 b0) 23)
                          b0 (mod64- b0 b1)
                          b3 (ror64 (logxor b3 b2) 40)
                          b2 (mod64- b2 b3)
                          b3 (ror64 (logxor b3 b0) 52)
                          b0 (mod64- b0 b3)
                          b1 (ror64 (logxor b1 b2) 57)
                          b2 (mod64- b2 b1)
                          b1 (ror64 (logxor b1 b0) 14)
                          b0 (mod64- b0 (mod64+ b1 ,x7))
                          b1 (mod64- b1 (mod64+ ,x8 ,x9))
                          b3 (ror64 (logxor b3 b2) 16)
                          b2 (mod64- b2 (mod64+ b3 (mod64+ ,x10 ,x11)))
                          b3 (mod64- b3 (mod64+ ,x12 ,x13)))))
        (arx k2 k3 t2 k4 t0 k0 17
             k1 k2 t1 k3 t2 k4 16)
        (arx k0 k1 t0 k2 t1 k3 15
             k4 k0 t2 k1 t0 k2 14)
        (arx k3 k4 t1 k0 t2 k1 13
             k2 k3 t0 k4 t1 k0 12)
        (arx k1 k2 t2 k3 t0 k4 11
             k0 k1 t1 k2 t2 k3 10)
        (arx k4 k0 t0 k1 t1 k2 9
             k3 k4 t2 k0 t0 k1 8)
        (arx k2 k3 t1 k4 t2 k0 7
             k1 k2 t0 k3 t1 k4 6)
        (arx k0 k1 t2 k2 t0 k3 5
             k4 k0 t1 k1 t2 k2 4)
        (arx k3 k4 t0 k0 t1 k1 3
             k2 k3 t2 k4 t0 k0 2)
        (arx k1 k2 t1 k3 t2 k4 1
             k0 k1 t0 k2 t1 k3 0))
      (store-words plaintext plaintext-start b0 b1 b2 b3)
      (values))))

(defcipher threefish256
  (:encrypt-function threefish256-encrypt-block)
  (:decrypt-function threefish256-decrypt-block)
  (:block-length 32)
  (:key-length (:fixed 32)))


;;; Implementation for blocks of 512 bits

(defclass threefish512 (cipher 64-byte-block-mixin)
  ((key :accessor threefish-key
        :type (simple-array (unsigned-byte 64) (9)))
   (tweak :accessor threefish-tweak
          :initform (make-array 3
                                :element-type '(unsigned-byte 64)
                                :initial-contents '(0 0 0))
          :type (simple-array (unsigned-byte 64) (3)))))

(defmethod shared-initialize :after ((cipher threefish512) slot-names
                                     &rest initargs
                                     &key (tweak nil tweak-p)
                                     &allow-other-keys)
  (when tweak
    (threefish-set-tweak cipher tweak))
  cipher)

(defmethod schedule-key ((cipher threefish512) key)
  (threefish-set-key cipher key)
  cipher)

(define-block-encryptor threefish512 64
  (let* ((key (threefish-key context))
         (tweak (threefish-tweak context))
         (k0 (aref key 0))
         (k1 (aref key 1))
         (k2 (aref key 2))
         (k3 (aref key 3))
         (k4 (aref key 4))
         (k5 (aref key 5))
         (k6 (aref key 6))
         (k7 (aref key 7))
         (k8 (aref key 8))
         (t0 (aref tweak 0))
         (t1 (aref tweak 1))
         (t2 (aref tweak 2)))
    (declare (type (simple-array (unsigned-byte 64) (9)) key)
             (type (simple-array (unsigned-byte 64) (3)) tweak)
             (type (unsigned-byte 64) k0 k1 k2 k3 k4 k5 k6 k7 k8 t0 t1 t2))
    (with-words ((b0 b1 b2 b3 b4 b5 b6 b7) plaintext plaintext-start :big-endian nil :size 8)
      (macrolet ((arx (x0 x1 x2 x3 x4 x5 x6 x7 x8 x9 x10
                       x11 x12 x13 x14 x15 x16 x17 x18 x19 x20 x21)
                   `(setf b1 (mod64+ b1 ,x0)
                          b0 (mod64+ b0 (mod64+ b1 ,x1))
                          b1 (logxor (rol64 b1 46) b0)
                          b3 (mod64+ b3 ,x2)
                          b2 (mod64+ b2 (mod64+ b3 ,x3))
                          b3 (logxor (rol64 b3 36) b2)
                          b5 (mod64+ b5 (mod64+ ,x4 ,x5))
                          b4 (mod64+ b4 (mod64+ b5 ,x6))
                          b5 (logxor (rol64 b5 19) b4)
                          b7 (mod64+ b7 (mod64+ ,x7 ,x8))
                          b6 (mod64+ b6 (mod64+ b7 (mod64+ ,x9 ,x10)))
                          b7 (logxor (rol64 b7 37) b6)
                          b2 (mod64+ b2 b1)
                          b1 (logxor (rol64 b1 33) b2)
                          b4 (mod64+ b4 b7)
                          b7 (logxor (rol64 b7 27) b4)
                          b6 (mod64+ b6 b5)
                          b5 (logxor (rol64 b5 14) b6)
                          b0 (mod64+ b0 b3)
                          b3 (logxor (rol64 b3 42) b0)
                          b4 (mod64+ b4 b1)
                          b1 (logxor (rol64 b1 17) b4)
                          b6 (mod64+ b6 b3)
                          b3 (logxor (rol64 b3 49) b6)
                          b0 (mod64+ b0 b5)
                          b5 (logxor (rol64 b5 36) b0)
                          b2 (mod64+ b2 b7)
                          b7 (logxor (rol64 b7 39) b2)
                          b6 (mod64+ b6 b1)
                          b1 (logxor (rol64 b1 44) b6)
                          b0 (mod64+ b0 b7)
                          b7 (logxor (rol64 b7 9) b0)
                          b2 (mod64+ b2 b5)
                          b5 (logxor (rol64 b5 54) b2)
                          b4 (mod64+ b4 b3)
                          b3 (logxor (rol64 b3 56) b4)
                          b1 (mod64+ b1 ,x11)
                          b0 (mod64+ b0 (mod64+ b1 ,x12))
                          b1 (logxor (rol64 b1 39) b0)
                          b3 (mod64+ b3 ,x13)
                          b2 (mod64+ b2 (mod64+ b3 ,x14))
                          b3 (logxor (rol64 b3 30) b2)
                          b5 (mod64+ b5 (mod64+ ,x15 ,x16))
                          b4 (mod64+ b4 (mod64+ b5 ,x17))
                          b5 (logxor (rol64 b5 34) b4)
                          b7 (mod64+ b7 (mod64+ ,x18 ,x19))
                          b6 (mod64+ b6 (mod64+ b7 (mod64+ ,x20 ,x21)))
                          b7 (logxor (rol64 b7 24) b6)
                          b2 (mod64+ b2 b1)
                          b1 (logxor (rol64 b1 13) b2)
                          b4 (mod64+ b4 b7)
                          b7 (logxor (rol64 b7 50) b4)
                          b6 (mod64+ b6 b5)
                          b5 (logxor (rol64 b5 10) b6)
                          b0 (mod64+ b0 b3)
                          b3 (logxor (rol64 b3 17) b0)
                          b4 (mod64+ b4 b1)
                          b1 (logxor (rol64 b1 25) b4)
                          b6 (mod64+ b6 b3)
                          b3 (logxor (rol64 b3 29) b6)
                          b0 (mod64+ b0 b5)
                          b5 (logxor (rol64 b5 39) b0)
                          b2 (mod64+ b2 b7)
                          b7 (logxor (rol64 b7 43) b2)
                          b6 (mod64+ b6 b1)
                          b1 (logxor (rol64 b1 8) b6)
                          b0 (mod64+ b0 b7)
                          b7 (logxor (rol64 b7 35) b0)
                          b2 (mod64+ b2 b5)
                          b5 (logxor (rol64 b5 56) b2)
                          b4 (mod64+ b4 b3)
                          b3 (logxor (rol64 b3 22) b4))))
        (arx k1 k0 k3 k2 k5 t0 k4 k7 0 k6 t1
             k2 k1 k4 k3 k6 t1 k5 k8 1 k7 t2)
        (arx k3 k2 k5 k4 k7 t2 k6 k0 2 k8 t0
             k4 k3 k6 k5 k8 t0 k7 k1 3 k0 t1)
        (arx k5 k4 k7 k6 k0 t1 k8 k2 4 k1 t2
             k6 k5 k8 k7 k1 t2 k0 k3 5 k2 t0)
        (arx k7 k6 k0 k8 k2 t0 k1 k4 6 k3 t1
             k8 k7 k1 k0 k3 t1 k2 k5 7 k4 t2)
        (arx k0 k8 k2 k1 k4 t2 k3 k6 8 k5 t0
             k1 k0 k3 k2 k5 t0 k4 k7 9 k6 t1)
        (arx k2 k1 k4 k3 k6 t1 k5 k8 10 k7 t2
             k3 k2 k5 k4 k7 t2 k6 k0 11 k8 t0)
        (arx k4 k3 k6 k5 k8 t0 k7 k1 12 k0 t1
             k5 k4 k7 k6 k0 t1 k8 k2 13 k1 t2)
        (arx k6 k5 k8 k7 k1 t2 k0 k3 14 k2 t0
             k7 k6 k0 k8 k2 t0 k1 k4 15 k3 t1)
        (arx k8 k7 k1 k0 k3 t1 k2 k5 16 k4 t2
             k0 k8 k2 k1 k4 t2 k3 k6 17 k5 t0))
      (setf b0 (mod64+ b0 k0)
            b1 (mod64+ b1 k1)
            b2 (mod64+ b2 k2)
            b3 (mod64+ b3 k3)
            b4 (mod64+ b4 k4)
            b5 (mod64+ b5 (mod64+ k5 t0))
            b6 (mod64+ b6 (mod64+ k6 t1))
            b7 (mod64+ b7 (mod64+ k7 18)))
      (store-words ciphertext ciphertext-start b0 b1 b2 b3 b4 b5 b6 b7)
      (values))))

(define-block-decryptor threefish512 64
  (let* ((key (threefish-key context))
         (tweak (threefish-tweak context))
         (k0 (aref key 0))
         (k1 (aref key 1))
         (k2 (aref key 2))
         (k3 (aref key 3))
         (k4 (aref key 4))
         (k5 (aref key 5))
         (k6 (aref key 6))
         (k7 (aref key 7))
         (k8 (aref key 8))
         (t0 (aref tweak 0))
         (t1 (aref tweak 1))
         (t2 (aref tweak 2)))
    (declare (type (simple-array (unsigned-byte 64) (9)) key)
             (type (simple-array (unsigned-byte 64) (3)) tweak)
             (type (unsigned-byte 64) k0 k1 k2 k3 k4 k5 k6 k7 k8 t0 t1 t2))
    (with-words ((b0 b1 b2 b3 b4 b5 b6 b7) ciphertext ciphertext-start :big-endian nil :size 8)
      (setf b0 (mod64- b0 k0)
            b1 (mod64- b1 k1)
            b2 (mod64- b2 k2)
            b3 (mod64- b3 k3)
            b4 (mod64- b4 k4)
            b5 (mod64- b5 (mod64+ k5 t0))
            b6 (mod64- b6 (mod64+ k6 t1))
            b7 (mod64- b7 (mod64+ k7 18)))
      (macrolet ((arx (x0 x1 x2 x3 x4 x5 x6 x7 x8 x9 x10
                       x11 x12 x13 x14 x15 x16 x17 x18 x19 x20 x21)
                   `(setf b3 (ror64 (logxor b3 b4) 22)
                          b4 (mod64- b4 b3)
                          b5 (ror64 (logxor b5 b2) 56)
                          b2 (mod64- b2 b5)
                          b7 (ror64 (logxor b7 b0) 35)
                          b0 (mod64- b0 b7)
                          b1 (ror64 (logxor b1 b6) 8)
                          b6 (mod64- b6 b1)
                          b7 (ror64 (logxor b7 b2) 43)
                          b2 (mod64- b2 b7)
                          b5 (ror64 (logxor b5 b0) 39)
                          b0 (mod64- b0 b5)
                          b3 (ror64 (logxor b3 b6) 29)
                          b6 (mod64- b6 b3)
                          b1 (ror64 (logxor b1 b4) 25)
                          b4 (mod64- b4 b1)
                          b3 (ror64 (logxor b3 b0) 17)
                          b0 (mod64- b0 b3)
                          b5 (ror64 (logxor b5 b6) 10)
                          b6 (mod64- b6 b5)
                          b7 (ror64 (logxor b7 b4) 50)
                          b4 (mod64- b4 b7)
                          b1 (ror64 (logxor b1 b2) 13)
                          b2 (mod64- b2 b1)
                          b7 (ror64 (logxor b7 b6) 24)
                          b6 (mod64- b6 (mod64+ b7 (mod64+ ,x0 ,x1)))
                          b7 (mod64- b7 (mod64+ ,x2 ,x3))
                          b5 (ror64 (logxor b5 b4) 34)
                          b4 (mod64- b4 (mod64+ b5 ,x4))
                          b5 (mod64- b5 (mod64+ ,x5 ,x6))
                          b3 (ror64 (logxor b3 b2) 30)
                          b2 (mod64- b2 (mod64+ b3 ,x7))
                          b3 (mod64- b3 ,x8)
                          b1 (ror64 (logxor b1 b0) 39)
                          b0 (mod64- b0 (mod64+ b1 ,x9))
                          b1 (mod64- b1 ,x10)
                          b3 (ror64 (logxor b3 b4) 56)
                          b4 (mod64- b4 b3)
                          b5 (ror64 (logxor b5 b2) 54)
                          b2 (mod64- b2 b5)
                          b7 (ror64 (logxor b7 b0) 9)
                          b0 (mod64- b0 b7)
                          b1 (ror64 (logxor b1 b6) 44)
                          b6 (mod64- b6 b1)
                          b7 (ror64 (logxor b7 b2) 39)
                          b2 (mod64- b2 b7)
                          b5 (ror64 (logxor b5 b0) 36)
                          b0 (mod64- b0 b5)
                          b3 (ror64 (logxor b3 b6) 49)
                          b6 (mod64- b6 b3)
                          b1 (ror64 (logxor b1 b4) 17)
                          b4 (mod64- b4 b1)
                          b3 (ror64 (logxor b3 b0) 42)
                          b0 (mod64- b0 b3)
                          b5 (ror64 (logxor b5 b6) 14)
                          b6 (mod64- b6 b5)
                          b7 (ror64 (logxor b7 b4) 27)
                          b4 (mod64- b4 b7)
                          b1 (ror64 (logxor b1 b2) 33)
                          b2 (mod64- b2 b1)
                          b7 (ror64 (logxor b7 b6) 37)
                          b6 (mod64- b6 (mod64+ b7 (mod64+ ,x11 ,x12)))
                          b7 (mod64- b7 (mod64+ ,x13 ,x14))
                          b5 (ror64 (logxor b5 b4) 19)
                          b4 (mod64- b4 (mod64+ b5 ,x15))
                          b5 (mod64- b5 (mod64+ ,x16 ,x17))
                          b3 (ror64 (logxor b3 b2) 36)
                          b2 (mod64- b2 (mod64+ b3 ,x18))
                          b3 (mod64- b3 ,x19)
                          b1 (ror64 (logxor b1 b0) 46)
                          b0 (mod64- b0 (mod64+ b1 ,x20))
                          b1 (mod64- b1 ,x21))))
        (arx k5 t0 k6 17 k3 k4 t2 k1 k2 k8 k0
             k4 t2 k5 16 k2 k3 t1 k0 k1 k7 k8)
        (arx k3 t1 k4 15 k1 k2 t0 k8 k0 k6 k7
             k2 t0 k3 14 k0 k1 t2 k7 k8 k5 k6)
        (arx k1 t2 k2 13 k8 k0 t1 k6 k7 k4 k5
             k0 t1 k1 12 k7 k8 t0 k5 k6 k3 k4)
        (arx k8 t0 k0 11 k6 k7 t2 k4 k5 k2 k3
             k7 t2 k8 10 k5 k6 t1 k3 k4 k1 k2)
        (arx k6 t1 k7 9 k4 k5 t0 k2 k3 k0 k1
             k5 t0 k6 8 k3 k4 t2 k1 k2 k8 k0)
        (arx k4 t2 k5 7 k2 k3 t1 k0 k1 k7 k8
             k3 t1 k4 6 k1 k2 t0 k8 k0 k6 k7)
        (arx k2 t0 k3 5 k0 k1 t2 k7 k8 k5 k6
             k1 t2 k2 4 k8 k0 t1 k6 k7 k4 k5)
        (arx k0 t1 k1 3 k7 k8 t0 k5 k6 k3 k4
             k8 t0 k0 2 k6 k7 t2 k4 k5 k2 k3)
        (arx k7 t2 k8 1 k5 k6 t1 k3 k4 k1 k2
             k6 t1 k7 0 k4 k5 t0 k2 k3 k0 k1))
      (store-words plaintext plaintext-start b0 b1 b2 b3 b4 b5 b6 b7)
      (values))))

(defcipher threefish512
  (:encrypt-function threefish512-encrypt-block)
  (:decrypt-function threefish512-decrypt-block)
  (:block-length 64)
  (:key-length (:fixed 64)))


;;; Implementation for blocks of 1024 bits

(defclass threefish1024 (cipher 128-byte-block-mixin)
  ((key :accessor threefish-key
        :type (simple-array (unsigned-byte 64) (17)))
   (tweak :accessor threefish-tweak
          :initform (make-array 3
                                :element-type '(unsigned-byte 64)
                                :initial-contents '(0 0 0))
          :type (simple-array (unsigned-byte 64) (3)))))

(defmethod shared-initialize :after ((cipher threefish1024) slot-names
                                     &rest initargs
                                     &key (tweak nil tweak-p)
                                     &allow-other-keys)
  (when tweak
    (threefish-set-tweak cipher tweak))
  cipher)

(defmethod schedule-key ((cipher threefish1024) key)
  (threefish-set-key cipher key)
  cipher)

(define-block-encryptor threefish1024 128
  (let* ((key (threefish-key context))
         (tweak (threefish-tweak context))
         (k0 (aref key 0))
         (k1 (aref key 1))
         (k2 (aref key 2))
         (k3 (aref key 3))
         (k4 (aref key 4))
         (k5 (aref key 5))
         (k6 (aref key 6))
         (k7 (aref key 7))
         (k8 (aref key 8))
         (k9 (aref key 9))
         (k10 (aref key 10))
         (k11 (aref key 11))
         (k12 (aref key 12))
         (k13 (aref key 13))
         (k14 (aref key 14))
         (k15 (aref key 15))
         (k16 (aref key 16))
         (t0 (aref tweak 0))
         (t1 (aref tweak 1))
         (t2 (aref tweak 2)))
    (declare (type (simple-array (unsigned-byte 64) (17)) key)
             (type (simple-array (unsigned-byte 64) (3)) tweak)
             (type (unsigned-byte 64) k0 k1 k2 k3 k4 k5 k6 k7 k8 k9 k10 k11 k12 k13 k14 k15 k16 t0 t1 t2))
    (with-words ((b0 b1 b2 b3 b4 b5 b6 b7 b8 b9 b10 b11 b12 b13 b14 b15) plaintext plaintext-start :big-endian nil :size 8)
      (macrolet ((arx (x0 x1 x2 x3 x4 x5 x6 x7 x8 x9 x10 x11 x12 x13 x14 x15 x16 x17 x18
                       x19 x20 x21 x22 x23 x24 x25 x26 x27 x28 x29 x30 x31 x32 x33 x34 x35 x36 x37)
                   `(setf b1 (mod64+ b1 ,x0)
                          b0 (mod64+ b0 (mod64+ b1 ,x1))
                          b1 (logxor (rol64 b1 24) b0)
                          b3 (mod64+ b3 ,x2)
                          b2 (mod64+ b2 (mod64+ b3 ,x3))
                          b3 (logxor (rol64 b3 13) b2)
                          b5 (mod64+ b5 ,x4)
                          b4 (mod64+ b4 (mod64+ b5 ,x5))
                          b5 (logxor (rol64 b5 8) b4)
                          b7 (mod64+ b7 ,x6)
                          b6 (mod64+ b6 (mod64+ b7 ,x7))
                          b7 (logxor (rol64 b7 47) b6)
                          b9 (mod64+ b9 ,x8)
                          b8 (mod64+ b8 (mod64+ b9 ,x9))
                          b9 (logxor (rol64 b9 8) b8)
                          b11 (mod64+ b11 ,x10)
                          b10 (mod64+ b10 (mod64+ b11 ,x11))
                          b11 (logxor (rol64 b11 17) b10)
                          b13 (mod64+ b13 (mod64+ ,x12 ,x13))
                          b12 (mod64+ b12 (mod64+ b13 ,x14))
                          b13 (logxor (rol64 b13 22) b12)
                          b15 (mod64+ b15 (mod64+ ,x15 ,x16))
                          b14 (mod64+ b14 (mod64+ b15 (mod64+ ,x17 ,x18)))
                          b15 (logxor (rol64 b15 37) b14)
                          b0 (mod64+ b0 b9)
                          b9 (logxor (rol64 b9 38) b0)
                          b2 (mod64+ b2 b13)
                          b13 (logxor (rol64 b13 19) b2)
                          b6 (mod64+ b6 b11)
                          b11 (logxor (rol64 b11 10) b6)
                          b4 (mod64+ b4 b15)
                          b15 (logxor (rol64 b15 55) b4)
                          b10 (mod64+ b10 b7)
                          b7 (logxor (rol64 b7 49) b10)
                          b12 (mod64+ b12 b3)
                          b3 (logxor (rol64 b3 18) b12)
                          b14 (mod64+ b14 b5)
                          b5 (logxor (rol64 b5 23) b14)
                          b8 (mod64+ b8 b1)
                          b1 (logxor (rol64 b1 52) b8)
                          b0 (mod64+ b0 b7)
                          b7 (logxor (rol64 b7 33) b0)
                          b2 (mod64+ b2 b5)
                          b5 (logxor (rol64 b5 4) b2)
                          b4 (mod64+ b4 b3)
                          b3 (logxor (rol64 b3 51) b4)
                          b6 (mod64+ b6 b1)
                          b1 (logxor (rol64 b1 13) b6)
                          b12 (mod64+ b12 b15)
                          b15 (logxor (rol64 b15 34) b12)
                          b14 (mod64+ b14 b13)
                          b13 (logxor (rol64 b13 41) b14)
                          b8 (mod64+ b8 b11)
                          b11 (logxor (rol64 b11 59) b8)
                          b10 (mod64+ b10 b9)
                          b9 (logxor (rol64 b9 17) b10)
                          b0 (mod64+ b0 b15)
                          b15 (logxor (rol64 b15 5) b0)
                          b2 (mod64+ b2 b11)
                          b11 (logxor (rol64 b11 20) b2)
                          b6 (mod64+ b6 b13)
                          b13 (logxor (rol64 b13 48) b6)
                          b4 (mod64+ b4 b9)
                          b9 (logxor (rol64 b9 41) b4)
                          b14 (mod64+ b14 b1)
                          b1 (logxor (rol64 b1 47) b14)
                          b8 (mod64+ b8 b5)
                          b5 (logxor (rol64 b5 28) b8)
                          b10 (mod64+ b10 b3)
                          b3 (logxor (rol64 b3 16) b10)
                          b12 (mod64+ b12 b7)
                          b7 (logxor (rol64 b7 25) b12)
                          b1 (mod64+ b1 ,x19)
                          b0 (mod64+ b0 (mod64+ b1 ,x20))
                          b1 (logxor (rol64 b1 41) b0)
                          b3 (mod64+ b3 ,x21)
                          b2 (mod64+ b2 (mod64+ b3 ,x22))
                          b3 (logxor (rol64 b3 9) b2)
                          b5 (mod64+ b5 ,x23)
                          b4 (mod64+ b4 (mod64+ b5 ,x24))
                          b5 (logxor (rol64 b5 37) b4)
                          b7 (mod64+ b7 ,x25)
                          b6 (mod64+ b6 (mod64+ b7 ,x26))
                          b7 (logxor (rol64 b7 31) b6)
                          b9 (mod64+ b9 ,x27)
                          b8 (mod64+ b8 (mod64+ b9 ,x28))
                          b9 (logxor (rol64 b9 12) b8)
                          b11 (mod64+ b11 ,x29)
                          b10 (mod64+ b10 (mod64+ b11 ,x30))
                          b11 (logxor (rol64 b11 47) b10)
                          b13 (mod64+ b13 (mod64+ ,x31 ,x32))
                          b12 (mod64+ b12 (mod64+ b13 ,x33))
                          b13 (logxor (rol64 b13 44) b12)
                          b15 (mod64+ b15 (mod64+ ,x34 ,x35))
                          b14 (mod64+ b14 (mod64+ b15 (mod64+ ,x36 ,x37)))
                          b15 (logxor (rol64 b15 30) b14)
                          b0 (mod64+ b0 b9)
                          b9 (logxor (rol64 b9 16) b0)
                          b2 (mod64+ b2 b13)
                          b13 (logxor (rol64 b13 34) b2)
                          b6 (mod64+ b6 b11)
                          b11 (logxor (rol64 b11 56) b6)
                          b4 (mod64+ b4 b15)
                          b15 (logxor (rol64 b15 51) b4)
                          b10 (mod64+ b10 b7)
                          b7 (logxor (rol64 b7 4) b10)
                          b12 (mod64+ b12 b3)
                          b3 (logxor (rol64 b3 53) b12)
                          b14 (mod64+ b14 b5)
                          b5 (logxor (rol64 b5 42) b14)
                          b8 (mod64+ b8 b1)
                          b1 (logxor (rol64 b1 41) b8)
                          b0 (mod64+ b0 b7)
                          b7 (logxor (rol64 b7 31) b0)
                          b2 (mod64+ b2 b5)
                          b5 (logxor (rol64 b5 44) b2)
                          b4 (mod64+ b4 b3)
                          b3 (logxor (rol64 b3 47) b4)
                          b6 (mod64+ b6 b1)
                          b1 (logxor (rol64 b1 46) b6)
                          b12 (mod64+ b12 b15)
                          b15 (logxor (rol64 b15 19) b12)
                          b14 (mod64+ b14 b13)
                          b13 (logxor (rol64 b13 42) b14)
                          b8 (mod64+ b8 b11)
                          b11 (logxor (rol64 b11 44) b8)
                          b10 (mod64+ b10 b9)
                          b9 (logxor (rol64 b9 25) b10)
                          b0 (mod64+ b0 b15)
                          b15 (logxor (rol64 b15 9) b0)
                          b2 (mod64+ b2 b11)
                          b11 (logxor (rol64 b11 48) b2)
                          b6 (mod64+ b6 b13)
                          b13 (logxor (rol64 b13 35) b6)
                          b4 (mod64+ b4 b9)
                          b9 (logxor (rol64 b9 52) b4)
                          b14 (mod64+ b14 b1)
                          b1 (logxor (rol64 b1 23) b14)
                          b8 (mod64+ b8 b5)
                          b5 (logxor (rol64 b5 31) b8)
                          b10 (mod64+ b10 b3)
                          b3 (logxor (rol64 b3 37) b10)
                          b12 (mod64+ b12 b7)
                          b7 (logxor (rol64 b7 20) b12))))
        (arx k1 k0 k3 k2 k5 k4 k7 k6 k9 k8 k11 k10 k13 t0 k12 k15 0 k14 t1
             k2 k1 k4 k3 k6 k5 k8 k7 k10 k9 k12 k11 k14 t1 k13 k16 1 k15 t2)
        (arx k3 k2 k5 k4 k7 k6 k9 k8 k11 k10 k13 k12 k15 t2 k14 k0 2 k16 t0
             k4 k3 k6 k5 k8 k7 k10 k9 k12 k11 k14 k13 k16 t0 k15 k1 3 k0 t1)
        (arx k5 k4 k7 k6 k9 k8 k11 k10 k13 k12 k15 k14 k0 t1 k16 k2 4 k1 t2
             k6 k5 k8 k7 k10 k9 k12 k11 k14 k13 k16 k15 k1 t2 k0 k3 5 k2 t0)
        (arx k7 k6 k9 k8 k11 k10 k13 k12 k15 k14 k0 k16 k2 t0 k1 k4 6 k3 t1
             k8 k7 k10 k9 k12 k11 k14 k13 k16 k15 k1 k0 k3 t1 k2 k5 7 k4 t2)
        (arx k9 k8 k11 k10 k13 k12 k15 k14 k0 k16 k2 k1 k4 t2 k3 k6 8 k5 t0
             k10 k9 k12 k11 k14 k13 k16 k15 k1 k0 k3 k2 k5 t0 k4 k7 9 k6 t1)
        (arx k11 k10 k13 k12 k15 k14 k0 k16 k2 k1 k4 k3 k6 t1 k5 k8 10 k7 t2
             k12 k11 k14 k13 k16 k15 k1 k0 k3 k2 k5 k4 k7 t2 k6 k9 11 k8 t0)
        (arx k13 k12 k15 k14 k0 k16 k2 k1 k4 k3 k6 k5 k8 t0 k7 k10 12 k9 t1
             k14 k13 k16 k15 k1 k0 k3 k2 k5 k4 k7 k6 k9 t1 k8 k11 13 k10 t2)
        (arx k15 k14 k0 k16 k2 k1 k4 k3 k6 k5 k8 k7 k10 t2 k9 k12 14 k11 t0
             k16 k15 k1 k0 k3 k2 k5 k4 k7 k6 k9 k8 k11 t0 k10 k13 15 k12 t1)
        (arx k0 k16 k2 k1 k4 k3 k6 k5 k8 k7 k10 k9 k12 t1 k11 k14 16 k13 t2
             k1 k0 k3 k2 k5 k4 k7 k6 k9 k8 k11 k10 k13 t2 k12 k15 17 k14 t0)
        (arx k2 k1 k4 k3 k6 k5 k8 k7 k10 k9 k12 k11 k14 t0 k13 k16 18 k15 t1
             k3 k2 k5 k4 k7 k6 k9 k8 k11 k10 k13 k12 k15 t1 k14 k0 19 k16 t2))
      (setf b0 (mod64+ b0 k3)
            b1 (mod64+ b1 k4)
            b2 (mod64+ b2 k5)
            b3 (mod64+ b3 k6)
            b4 (mod64+ b4 k7)
            b5 (mod64+ b5 k8)
            b6 (mod64+ b6 k9)
            b7 (mod64+ b7 k10)
            b8 (mod64+ b8 k11)
            b9 (mod64+ b9 k12)
            b10 (mod64+ b10 k13)
            b11 (mod64+ b11 k14)
            b12 (mod64+ b12 k15)
            b13 (mod64+ b13 (mod64+ k16 t2))
            b14 (mod64+ b14 (mod64+ k0 t0))
            b15 (mod64+ b15 (mod64+ k1 20)))
      (store-words ciphertext ciphertext-start b0 b1 b2 b3 b4 b5 b6 b7 b8 b9 b10 b11 b12 b13 b14 b15)
      (values))))

(define-block-decryptor threefish1024 128
  (let* ((key (threefish-key context))
         (tweak (threefish-tweak context))
         (k0 (aref key 0))
         (k1 (aref key 1))
         (k2 (aref key 2))
         (k3 (aref key 3))
         (k4 (aref key 4))
         (k5 (aref key 5))
         (k6 (aref key 6))
         (k7 (aref key 7))
         (k8 (aref key 8))
         (k9 (aref key 9))
         (k10 (aref key 10))
         (k11 (aref key 11))
         (k12 (aref key 12))
         (k13 (aref key 13))
         (k14 (aref key 14))
         (k15 (aref key 15))
         (k16 (aref key 16))
         (t0 (aref tweak 0))
         (t1 (aref tweak 1))
         (t2 (aref tweak 2)))
    (declare (type (simple-array (unsigned-byte 64) (17)) key)
             (type (simple-array (unsigned-byte 64) (3)) tweak)
             (type (unsigned-byte 64) k0 k1 k2 k3 k4 k5 k6 k7 k8 k9 k10 k11 k12 k13 k14 k15 k16 t0 t1 t2))
    (with-words ((b0 b1 b2 b3 b4 b5 b6 b7 b8 b9 b10 b11 b12 b13 b14 b15) ciphertext ciphertext-start :big-endian nil :size 8)
      (setf b0 (mod64- b0 k3)
            b1 (mod64- b1 k4)
            b2 (mod64- b2 k5)
            b3 (mod64- b3 k6)
            b4 (mod64- b4 k7)
            b5 (mod64- b5 k8)
            b6 (mod64- b6 k9)
            b7 (mod64- b7 k10)
            b8 (mod64- b8 k11)
            b9 (mod64- b9 k12)
            b10 (mod64- b10 k13)
            b11 (mod64- b11 k14)
            b12 (mod64- b12 k15)
            b13 (mod64- b13 (mod64+ k16 t2))
            b14 (mod64- b14 (mod64+ k0 t0))
            b15 (mod64- b15 (mod64+ k1 20)))
      (macrolet ((arx (x0 x1 x2 x3 x4 x5 x6 x7 x8 x9 x10 x11 x12 x13 x14 x15 x16 x17 x18
                       x19 x20 x21 x22 x23 x24 x25 x26 x27 x28 x29 x30 x31 x32 x33 x34 x35 x36 x37)
                   `(setf b7 (ror64 (logxor b7 b12) 20)
                          b12 (mod64- b12 b7)
                          b3 (ror64 (logxor b3 b10) 37)
                          b10 (mod64- b10 b3)
                          b5 (ror64 (logxor b5 b8) 31)
                          b8 (mod64- b8 b5)
                          b1 (ror64 (logxor b1 b14) 23)
                          b14 (mod64- b14 b1)
                          b9 (ror64 (logxor b9 b4) 52)
                          b4 (mod64- b4 b9)
                          b13 (ror64 (logxor b13 b6) 35)
                          b6 (mod64- b6 b13)
                          b11 (ror64 (logxor b11 b2) 48)
                          b2 (mod64- b2 b11)
                          b15 (ror64 (logxor b15 b0) 9)
                          b0 (mod64- b0 b15)
                          b9 (ror64 (logxor b9 b10) 25)
                          b10 (mod64- b10 b9)
                          b11 (ror64 (logxor b11 b8) 44)
                          b8 (mod64- b8 b11)
                          b13 (ror64 (logxor b13 b14) 42)
                          b14 (mod64- b14 b13)
                          b15 (ror64 (logxor b15 b12) 19)
                          b12 (mod64- b12 b15)
                          b1 (ror64 (logxor b1 b6) 46)
                          b6 (mod64- b6 b1)
                          b3 (ror64 (logxor b3 b4) 47)
                          b4 (mod64- b4 b3)
                          b5 (ror64 (logxor b5 b2) 44)
                          b2 (mod64- b2 b5)
                          b7 (ror64 (logxor b7 b0) 31)
                          b0 (mod64- b0 b7)
                          b1 (ror64 (logxor b1 b8) 41)
                          b8 (mod64- b8 b1)
                          b5 (ror64 (logxor b5 b14) 42)
                          b14 (mod64- b14 b5)
                          b3 (ror64 (logxor b3 b12) 53)
                          b12 (mod64- b12 b3)
                          b7 (ror64 (logxor b7 b10) 4)
                          b10 (mod64- b10 b7)
                          b15 (ror64 (logxor b15 b4) 51)
                          b4 (mod64- b4 b15)
                          b11 (ror64 (logxor b11 b6) 56)
                          b6 (mod64- b6 b11)
                          b13 (ror64 (logxor b13 b2) 34)
                          b2 (mod64- b2 b13)
                          b9 (ror64 (logxor b9 b0) 16)
                          b0 (mod64- b0 b9)
                          b15 (ror64 (logxor b15 b14) 30)
                          b14 (mod64- b14 (mod64+ b15 (mod64+ ,x0 ,x1)))
                          b15 (mod64- b15 (mod64+ ,x2 ,x3))
                          b13 (ror64 (logxor b13 b12) 44)
                          b12 (mod64- b12 (mod64+ b13 ,x4))
                          b13 (mod64- b13 (mod64+ ,x5 ,x6))
                          b11 (ror64 (logxor b11 b10) 47)
                          b10 (mod64- b10 (mod64+ b11 ,x7))
                          b11 (mod64- b11 ,x8)
                          b9 (ror64 (logxor b9 b8) 12)
                          b8 (mod64- b8 (mod64+ b9 ,x9))
                          b9 (mod64- b9 ,x10)
                          b7 (ror64 (logxor b7 b6) 31)
                          b6 (mod64- b6 (mod64+ b7 ,x11))
                          b7 (mod64- b7 ,x12)
                          b5 (ror64 (logxor b5 b4) 37)
                          b4 (mod64- b4 (mod64+ b5 ,x13))
                          b5 (mod64- b5 ,x14)
                          b3 (ror64 (logxor b3 b2) 9)
                          b2 (mod64- b2 (mod64+ b3 ,x15))
                          b3 (mod64- b3 ,x16)
                          b1 (ror64 (logxor b1 b0) 41)
                          b0 (mod64- b0 (mod64+ b1 ,x17))
                          b1 (mod64- b1 ,x18)
                          b7 (ror64 (logxor b7 b12) 25)
                          b12 (mod64- b12 b7)
                          b3 (ror64 (logxor b3 b10) 16)
                          b10 (mod64- b10 b3)
                          b5 (ror64 (logxor b5 b8) 28)
                          b8 (mod64- b8 b5)
                          b1 (ror64 (logxor b1 b14) 47)
                          b14 (mod64- b14 b1)
                          b9 (ror64 (logxor b9 b4) 41)
                          b4 (mod64- b4 b9)
                          b13 (ror64 (logxor b13 b6) 48)
                          b6 (mod64- b6 b13)
                          b11 (ror64 (logxor b11 b2) 20)
                          b2 (mod64- b2 b11)
                          b15 (ror64 (logxor b0 b15) 5)
                          b0 (mod64- b0 b15)
                          b9 (ror64 (logxor b9 b10) 17)
                          b10 (mod64- b10 b9)
                          b11 (ror64 (logxor b11 b8) 59)
                          b8 (mod64- b8 b11)
                          b13 (ror64 (logxor b13 b14) 41)
                          b14 (mod64- b14 b13)
                          b15 (ror64 (logxor b15 b12) 34)
                          b12 (mod64- b12 b15)
                          b1 (ror64 (logxor b1 b6) 13)
                          b6 (mod64- b6 b1)
                          b3 (ror64 (logxor b3 b4) 51)
                          b4 (mod64- b4 b3)
                          b5 (ror64 (logxor b5 b2) 4)
                          b2 (mod64- b2 b5)
                          b7 (ror64 (logxor b7 b0) 33)
                          b0 (mod64- b0 b7)
                          b1 (ror64 (logxor b1 b8) 52)
                          b8 (mod64- b8 b1)
                          b5 (ror64 (logxor b5 b14) 23)
                          b14 (mod64- b14 b5)
                          b3 (ror64 (logxor b3 b12) 18)
                          b12 (mod64- b12 b3)
                          b7 (ror64 (logxor b7 b10) 49)
                          b10 (mod64- b10 b7)
                          b15 (ror64 (logxor b15 b4) 55)
                          b4 (mod64- b4 b15)
                          b11 (ror64 (logxor b11 b6) 10)
                          b6 (mod64- b6 b11)
                          b13 (ror64 (logxor b13 b2) 19)
                          b2 (mod64- b2 b13)
                          b9 (ror64 (logxor b9 b0) 38)
                          b0 (mod64- b0 b9)
                          b15 (ror64 (logxor b15 b14) 37)
                          b14 (mod64- b14 (mod64+ b15 (mod64+ ,x19 ,x20)))
                          b15 (mod64- b15 (mod64+ ,x21 ,x22))
                          b13 (ror64 (logxor b13 b12) 22)
                          b12 (mod64- b12 (mod64+ b13 ,x23))
                          b13 (mod64- b13 (mod64+ ,x24 ,x25))
                          b11 (ror64 (logxor b11 b10) 17)
                          b10 (mod64- b10 (mod64+ b11 ,x26))
                          b11 (mod64- b11 ,x27)
                          b9 (ror64 (logxor b9 b8) 8)
                          b8 (mod64- b8 (mod64+ b9 ,x28))
                          b9 (mod64- b9 ,x29)
                          b7 (ror64 (logxor b7 b6) 47)
                          b6 (mod64- b6 (mod64+ b7 ,x30))
                          b7 (mod64- b7 ,x31)
                          b5 (ror64 (logxor b5 b4) 8)
                          b4 (mod64- b4 (mod64+ b5 ,x32))
                          b5 (mod64- b5 ,x33)
                          b3 (ror64 (logxor b3 b2) 13)
                          b2 (mod64- b2 (mod64+ b3 ,x34))
                          b3 (mod64- b3 ,x35)
                          b1 (ror64 (logxor b1 b0) 24)
                          b0 (mod64- b0 (mod64+ b1 ,x36))
                          b1 (mod64- b1 ,x37))))
        (arx k16 t2 k0 19 k14 k15 t1 k12 k13 k10 k11 k8 k9 k6 k7 k4 k5 k2 k3
             k15 t1 k16 18 k13 k14 t0 k11 k12 k9 k10 k7 k8 k5 k6 k3 k4 k1 k2)
        (arx k14 t0 k15 17 k12 k13 t2 k10 k11 k8 k9 k6 k7 k4 k5 k2 k3 k0 k1
             k13 t2 k14 16 k11 k12 t1 k9 k10 k7 k8 k5 k6 k3 k4 k1 k2 k16 k0)
        (arx k12 t1 k13 15 k10 k11 t0 k8 k9 k6 k7 k4 k5 k2 k3 k0 k1 k15 k16
             k11 t0 k12 14 k9 k10 t2 k7 k8 k5 k6 k3 k4 k1 k2 k16 k0 k14 k15)
        (arx k10 t2 k11 13 k8 k9 t1 k6 k7 k4 k5 k2 k3 k0 k1 k15 k16 k13 k14
             k9 t1 k10 12 k7 k8 t0 k5 k6 k3 k4 k1 k2 k16 k0 k14 k15 k12 k13)
        (arx k8 t0 k9 11 k6 k7 t2 k4 k5 k2 k3 k0 k1 k15 k16 k13 k14 k11 k12
             k7 t2 k8 10 k5 k6 t1 k3 k4 k1 k2 k16 k0 k14 k15 k12 k13 k10 k11)
        (arx k6 t1 k7 9 k4 k5 t0 k2 k3 k0 k1 k15 k16 k13 k14 k11 k12 k9 k10
             k5 t0 k6 8 k3 k4 t2 k1 k2 k16 k0 k14 k15 k12 k13 k10 k11 k8 k9)
        (arx k4 t2 k5 7 k2 k3 t1 k0 k1 k15 k16 k13 k14 k11 k12 k9 k10 k7 k8
             k3 t1 k4 6 k1 k2 t0 k16 k0 k14 k15 k12 k13 k10 k11 k8 k9 k6 k7)
        (arx k2 t0 k3 5 k0 k1 t2 k15 k16 k13 k14 k11 k12 k9 k10 k7 k8 k5 k6
             k1 t2 k2 4 k16 k0 t1 k14 k15 k12 k13 k10 k11 k8 k9 k6 k7 k4 k5)
        (arx k0 t1 k1 3 k15 k16 t0 k13 k14 k11 k12 k9 k10 k7 k8 k5 k6 k3 k4
             k16 t0 k0 2 k14 k15 t2 k12 k13 k10 k11 k8 k9 k6 k7 k4 k5 k2 k3)
        (arx k15 t2 k16 1 k13 k14 t1 k11 k12 k9 k10 k7 k8 k5 k6 k3 k4 k1 k2
             k14 t1 k15 0 k12 k13 t0 k10 k11 k8 k9 k6 k7 k4 k5 k2 k3 k0 k1))
      (store-words plaintext plaintext-start b0 b1 b2 b3 b4 b5 b6 b7 b8 b9 b10 b11 b12 b13 b14 b15)
      (values))))

(defcipher threefish1024
  (:encrypt-function threefish1024-encrypt-block)
  (:decrypt-function threefish1024-decrypt-block)
  (:block-length 128)
  (:key-length (:fixed 128)))
